import { useState } from 'react';
import { MapView } from './components/MapView';
import { TopBar } from './components/TopBar';
import { InfoPanel } from './components/InfoPanel';
import { LoginModal } from './components/LoginModal';
import { SubmitEventForm } from './components/SubmitEventForm';
import { FeedbackModal } from './components/FeedbackModal';
import { RulesModal } from './components/RulesModal';
import { CommunityGovernanceModal } from './components/CommunityGovernanceModal';
import { eventsData, EventData } from './data/eventData';

interface Event {
  id: string;
  name: string;
  description: string;
  age: 'new' | 'old';
  verification: 'verified' | 'trusted' | 'unverified';
  position: { x: number; y: number };
  category: string;
  date: string;
}

// Helper function to convert EventData to Event based on language
const convertToEvent = (eventData: EventData, language: 'nl' | 'en'): Event => ({
  id: eventData.id,
  name: eventData.name[language],
  description: eventData.description[language],
  age: eventData.age,
  verification: eventData.verification,
  position: eventData.position,
  category: eventData.category[language],
  date: eventData.date
});

export default function App() {
  const [mapType, setMapType] = useState<'standard' | 'satellite' | 'transit'>('standard');
  const [ecoMode, setEcoMode] = useState(false);
  const [isLoggedIn, setIsLoggedIn] = useState(false);
  const [username, setUsername] = useState('');
  const [showLoginModal, setShowLoginModal] = useState(false);
  const [showSubmitEventForm, setShowSubmitEventForm] = useState(false);
  const [showFeedbackModal, setShowFeedbackModal] = useState(false);
  const [showRulesModal, setShowRulesModal] = useState(false);
  const [showCommunityGovernanceModal, setShowCommunityGovernanceModal] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [showTopBar, setShowTopBar] = useState(false);
  const [highlightedEventId, setHighlightedEventId] = useState<string | null>(null);
  const [selectedCategories, setSelectedCategories] = useState<string[]>([]);
  const [selectedAgeFilters, setSelectedAgeFilters] = useState<('new' | 'old')[]>(['new', 'old']);
  const [selectedVerificationFilters, setSelectedVerificationFilters] = useState<('verified' | 'trusted' | 'unverified')[]>(['verified', 'trusted', 'unverified']);
  const [language, setLanguage] = useState<'nl' | 'en'>('nl');
  const [customEvents, setCustomEvents] = useState<EventData[]>([]);

  // Convert events based on current language (combine predefined + custom events)
  const events = [...eventsData, ...customEvents].map(e => convertToEvent(e, language));

  const handleLogin = (username: string, password: string) => {
    // Simulate login
    setIsLoggedIn(true);
    setUsername(username);
    setShowLoginModal(false);
  };

  const handleLogout = () => {
    setIsLoggedIn(false);
    setUsername('');
  };

  const handleAddEvent = (eventData: { name: string; description: string; location: string; category: string; date: string; position: { x: number; y: number } }) => {
    const newEventData: EventData = {
      id: `user-${Date.now()}`,
      name: { nl: eventData.name, en: eventData.name }, // User events blijven in ingevoerde taal
      description: { nl: eventData.description, en: eventData.description },
      age: 'new',
      verification: 'unverified',
      position: eventData.position,
      category: { nl: eventData.category, en: eventData.category },
      date: eventData.date
    };

    setCustomEvents(prev => [...prev, newEventData]);
    setShowSubmitEventForm(false);
    
    // Highlight the new event for 10 seconds
    setHighlightedEventId(newEventData.id);
    setTimeout(() => {
      setHighlightedEventId(null);
    }, 10000);
  };

  const handleFeedbackSubmit = (feedback: { name: string; email: string; category: string; message: string }) => {
    // In een echte applicatie zou dit naar een backend API gestuurd worden
    console.log('Feedback ontvangen:', feedback);
    // Toon een bevestigingsbericht (optioneel: maak een toast notification)
    alert('Bedankt voor uw feedback! We hebben uw bericht ontvangen.');
  };

  return (
    <div className="w-full h-screen flex flex-col bg-gray-50">
      {showTopBar && (
        <TopBar
          mapType={mapType}
          setMapType={setMapType}
          ecoMode={ecoMode}
          setEcoMode={setEcoMode}
          isLoggedIn={isLoggedIn}
          username={username}
          onLoginClick={() => setShowLoginModal(true)}
          onLogoutClick={handleLogout}
          searchQuery={searchQuery}
          setSearchQuery={setSearchQuery}
          onSubmitEventClick={() => setShowSubmitEventForm(true)}
          onFeedbackClick={() => setShowFeedbackModal(true)}
          onRulesClick={() => setShowRulesModal(true)}
          onCommunityGovernanceClick={() => setShowCommunityGovernanceModal(true)}
          selectedCategories={selectedCategories}
          setSelectedCategories={setSelectedCategories}
          selectedAgeFilters={selectedAgeFilters}
          setSelectedAgeFilters={setSelectedAgeFilters}
          selectedVerificationFilters={selectedVerificationFilters}
          setSelectedVerificationFilters={setSelectedVerificationFilters}
          language={language}
          setLanguage={setLanguage}
        />
      )}
      
      <div className="flex-1 flex relative">
        <InfoPanel 
          mapType={mapType}
          onMoreFunctionsClick={() => setShowTopBar(!showTopBar)}
          isLoggedIn={isLoggedIn}
          username={username}
          onLoginClick={() => setShowLoginModal(true)}
          onLogoutClick={handleLogout}
          language={language}
        />
        
        <div className="flex-1 relative">
          <MapView
            mapType={mapType}
            ecoMode={ecoMode}
            isLoggedIn={isLoggedIn}
            searchQuery={searchQuery}
            events={events}
            highlightedEventId={highlightedEventId}
            selectedCategories={selectedCategories}
            selectedAgeFilters={selectedAgeFilters}
            selectedVerificationFilters={selectedVerificationFilters}
            language={language}
          />
        </div>
      </div>

      {showLoginModal && (
        <LoginModal
          onClose={() => setShowLoginModal(false)}
          onLogin={handleLogin}
          language={language}
        />
      )}

      {showSubmitEventForm && (
        <SubmitEventForm
          onClose={() => setShowSubmitEventForm(false)}
          onAddEvent={handleAddEvent}
        />
      )}

      {showFeedbackModal && (
        <FeedbackModal
          onClose={() => setShowFeedbackModal(false)}
          onSubmit={handleFeedbackSubmit}
        />
      )}

      {showRulesModal && (
        <RulesModal
          onClose={() => setShowRulesModal(false)}
          language={language}
        />
      )}

      {showCommunityGovernanceModal && (
        <CommunityGovernanceModal
          onClose={() => setShowCommunityGovernanceModal(false)}
          language={language}
        />
      )}
    </div>
  );
}